﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Globalization;
using System.Threading;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessManager.TBIInstrumentVIAsrv;
using System.Data.Linq;
using System.Web;
using System.Web.UI;
using System.Web.Configuration;

namespace VeteransAffairs.Registries.BusinessManager.TBI
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class VIAInstrumentsManager : BaseBO
    {
        #region Variable Declarations

        private TbiServiceInterfaceClient _VIAService;
        private personsTO _personsTO;
        private CprsManager _cprsManager = new CprsManager();

        MDWSLog mdwsLog = new MDWSLog();
        MDWS_LOG _mdwsLog = new MDWS_LOG();

        protected string _loginSiteCode = null;
        private string _userID = String.Empty;
        private string _tokenizedDUZ = String.Empty;
        private string _localPid = String.Empty;
        private string _savedItemID = String.Empty;
        private string _providerName = String.Empty;

        public TBIServiceTOVIA TbiServiceTo { get; set; } // 01-Nov-2010 jps  replace [internal set] with [set] per JH;

        #endregion

        #region Page Load Events

        public VIAInstrumentsManager()
        {
            _defaultSortField = "SURVEYS_ID";
            if (_loginSiteCode == null)
            {
                if (!Helpers.IsSessionVariableNullOrEmpty(HttpContext.Current.Session["SiteCode"]))
                    _loginSiteCode = HttpContext.Current.Session["SiteCode"].ToString();
                else
                    _loginSiteCode = "000";
            }

        }

        ~VIAInstrumentsManager()
        {
        }

        /// <summary>
        /// Method used to retrieve all "active" instruments from the database.  Replaces legacy systems usage of
        /// TbiServiceTo.Instruments (static) list.
        /// </summary>
        /// <returns></returns>
        public List<TBI.InstrumentType> GetActiveInstrumentList()
        {
            using (_db = GetDataContext())
            {
                int tbiRegistryId = _db.STD_REGISTRies.FirstOrDefault(R => R.CODE == "TBI").ID;
                return convertStdSurveyTypeListToInstrumentType(_db.STD_SURVEY_TYPEs.Where(S => S.REGISTRY_ID == tbiRegistryId && S.IS_ACTIVE == true).OrderBy(I => I.NAME).ToList());
            }
        }

        /// <summary>
        /// Method used to retrieve all instruments from the database.  Replaces legacy systems usage of
        /// TbiServiceTo.Instruments (static) list.
        /// </summary>
        /// <returns></returns>
        public List<TBI.InstrumentType> GetAllInstrumentList()
        {
            using (_db = GetDataContext())
            {
                int tbiRegistryId = _db.STD_REGISTRies.FirstOrDefault(R => R.CODE == "TBI").ID;
                return convertStdSurveyTypeListToInstrumentType(_db.STD_SURVEY_TYPEs.Where(S => S.REGISTRY_ID == tbiRegistryId).OrderBy(I => I.NAME).ToList());
            }
        }

        /// <summary>
        /// Set Instrument Active Status by ID
        /// </summary>
        /// <param name="surveyTypeId"></param>
        /// <param name="isActive"></param>
        public void SetInstrumentActiviationStatus(int surveyTypeId, bool isActive)
        {
            using (_db = GetDataContext())
            {
                STD_SURVEY_TYPE surveyType = _db.STD_SURVEY_TYPEs.FirstOrDefault(S => S.ID == surveyTypeId);
                if (surveyType != null)
                {
                    surveyType.IS_ACTIVE = isActive;

                    _db.SubmitChanges();
                }
            }
        }

        /// <summary>
        /// Returns the Instrument Status - is active.
        /// </summary>
        /// <param name="surveyTypeId"></param>
        /// <param name="isActive"></param>
        public bool IsInstrumentActive(int surveyTypeId)
        {
            using (_db = GetDataContext())
            {
                STD_SURVEY_TYPE surveyType = _db.STD_SURVEY_TYPEs.FirstOrDefault(S => S.ID == surveyTypeId);
                if (surveyType != null)
                {
                    return surveyType.IS_ACTIVE;
                }
            }

            return false;
        }

        public bool LoadTbi_VIAService(TBIServiceTOVIA tbiServiceTo)
        {
            //_VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");
            _VIAService = TBIInstrumentsManager.CreateVIAClientFromDB();
            if (_VIAService == null)
                _VIAService = new TbiServiceInterfaceClient("TbiServiceImplPort");
            if (HttpContext.Current.Session["ViaEndpoint"] == null)
            {
                Logging.WriteLogToFile("LoadTBI_VIAService - VIAIntrument Manager: Using VIA endpoint from config: " + _VIAService.Endpoint.Address.ToString());
                HttpContext.Current.Session["ViaEndpoint"] = _VIAService.Endpoint.Address.ToString();
            }

            this.TbiServiceTo = tbiServiceTo;

            _personsTO = CPRSLaunchVIA(tbiServiceTo);

            if (_personsTO == null)
            {
                this.TbiServiceTo = null;
            }

            if (this.TbiServiceTo != null)
            {
                if (_personsTO != null && _personsTO.patient != null)
                {
                    this.TbiServiceTo.StationNumber = _loginSiteCode;
                    this.TbiServiceTo.PatientName = _personsTO.patient.patientName;
                    this.TbiServiceTo.PatientDOB = _personsTO.patient.dob;
                    this.TbiServiceTo.PatientGender = _personsTO.patient.gender;
                    this.TbiServiceTo.PatientSSN = _personsTO.patient.ssn;
                    this.TbiServiceTo.PatientICN = _personsTO.patient.mpiPid;
                    this.TbiServiceTo.PatientDFN = _personsTO.patient.localPid;
                    this.TbiServiceTo.PatientAdmintTimeStamp = _personsTO.patient.admitTimestamp;
                }

                if (_personsTO.user != null)
                {
                    this.TbiServiceTo.STA6N = _personsTO.user.siteId;
                    this.TbiServiceTo.UserName = _personsTO.user.name;
                    _providerName = _personsTO.user.name;
                    _tokenizedDUZ = _personsTO.user.DUZ;

                    if (HttpContext.Current.Session["TokenizedDUZ"] == null)
                    {
                        HttpContext.Current.Session["TokenizedDUZ"] = _tokenizedDUZ;
                    }
                    if (HttpContext.Current.Session["ProviderName"] == null)
                    {
                        HttpContext.Current.Session["ProviderName"] = _providerName;
                    }
                }

                if (_personsTO.patient != null)
                {
                    if (_personsTO.patient.location != null)
                    {
                        this.TbiServiceTo.PatientLocationId = _personsTO.patient.location.id;

                        if (_personsTO.patient.location.facility != null)
                        {
                            this.TbiServiceTo.InstitutionName = _personsTO.patient.location.facility.name;
                        }
                    }
                }
            }

            if (this.TbiServiceTo == null)
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        #endregion

        #region VIA Service Reference Calls
        private personsTO CPRSLaunchVIA(TBIServiceTOVIA tbiServiceTo)
        {
            personsTO result = null;

            try
            {
                provider _providerbean = new provider();
                patient _patientbean = new patient();
                queryBean _bean = new queryBean();

                _userID = tbiServiceTo.DUZ;
                //_userID = "520736420";
                _localPid = tbiServiceTo.DFN;
                HttpContext.Current.Session["ProviderDFN"] = _localPid;

                _bean.criteria = "TBI_PWD";
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                //patient attributes
                _patientbean.localSiteId = _loginSiteCode;
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _bean.patient = _patientbean;
                //provider attributes
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = _userID;
                _bean.provider = _providerbean;

                result = _VIAService.cprsLaunch(_bean);

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "cprsLaunch-VIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { "TBI-PWD", _loginSiteCode, _userID, _localPid });

                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    _mdwsLog.ERROR_LEVEL = 0;//success
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 1;//failed service launch
                }

            }
            catch (Exception ex)
            {
                Helpers.HandleTBIException(ex, "VIAInstrumentManager.cs", "", "CPRSLaunchVIA()");
                string testerror = ex.Message.ToString();
                _mdwsLog.ERROR_LEVEL = 2;
                mdwsLog.LogMDWSCall(_mdwsLog);
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;

        }

        public List<KeyValuePair<string, string>> getNoteTitlesFromVIA(string target, string direction, out string error)
        {
            taggedTextArray ret = null;
            List<KeyValuePair<string, string>> result = null;

            string searchParameter = String.Empty;

            if (target.Trim() == String.Empty)
            {
                searchParameter = "T";//defaults to tbi note get
            }
            else
            {
                searchParameter = target.Trim().ToUpper();
            }

            try
            {
                if (_VIAService != null)
                {
                    queryBean _bean = new queryBean();
                    patient _patientbean = new patient();
                    provider _providerbean = new provider();

                    _loginSiteCode = this.TbiServiceTo.StationNumber;

                    _bean.target = searchParameter;
                    _bean.direction = "1";
                    _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                    _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                    _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                    _patientbean.localSiteId = _loginSiteCode;
                    _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                    _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                    _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                    _providerbean.loginSiteCode = _loginSiteCode;
                    _bean.patient = _patientbean;
                    _bean.provider = _providerbean;

                    ret = _VIAService.getNoteTitles(_bean);

                    _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                    _mdwsLog.SERVICE_NAME = "getNoteTitles-VIA";
                    _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { "TBI-PWD", _loginSiteCode, HttpContext.Current.Session["TokenizedDUZ"].ToString(), HttpContext.Current.Session["ProviderDFN"].ToString() });

                    if (ret != null)
                    {
                        if (ret.fault != null)
                        {
                            error = ret.fault.message;
                            result = null;

                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                        else
                        {
                            if (ret.count > 0)
                            {
                                _mdwsLog.ERROR_LEVEL = 0;
                                error = "SUCCESS";
                                result = (from e in ret.results.FirstOrDefault().taggedResults orderby e.text select new KeyValuePair<string, string>(e.tag, e.textArray[0])).ToList();
                                _mdwsLog.RETURNED_DATA = result.Serialize();
                            }
                            else
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                                error = null;
                            }
                        }
                    }
                    else
                    {
                        error = error = "VIA Service call get Note Titles returned null";
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                else
                {
                    error = null;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "getNoteTitlesFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;

        }
        public List<KeyValuePair<string, string>> getConsultsFromVIA(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedConsultArrays ret = null;

            try
            {
                if (_VIAService != null)
                {
                    queryBean _bean = new queryBean();
                    patient _patientbean = new patient();
                    provider _providerbean = new provider();

                    _loginSiteCode = this.TbiServiceTo.StationNumber;

                    _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                    _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                    _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                    _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                    _patientbean.localSiteId = _loginSiteCode;
                    _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                    _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                    _providerbean.loginSiteCode = _loginSiteCode;
                    _bean.patient = _patientbean;
                    _bean.provider = _providerbean;

                    ret = _VIAService.getConsultsForPatient(_bean);

                    _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                    _mdwsLog.SERVICE_NAME = "getConsultsFromVIA-VIA";
                    _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { "TBI-PWD", _loginSiteCode, HttpContext.Current.Session["TokenizedDUZ"].ToString(), HttpContext.Current.Session["ProviderDFN"].ToString() });

                    if (ret != null)
                    {
                        if (ret.fault != null)
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = ret.fault.message;
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            if (!(ret.count == 0 || ret.arrays[0].consults == null))
                            {
                                var consultsArray = ret.arrays.SelectMany(s => s.consults).ToList();
                                result = (from e in consultsArray orderby e.timestamp descending select new KeyValuePair<string, string>(e.id, DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + " (" + e.status + ") " + e.title + " " + "Consult #: " + e.id)).ToList();
                                _mdwsLog.RETURNED_DATA = result.Serialize();
                            }
                            else
                            {
                                _mdwsLog.ERROR_LEVEL = 1;
                                error = null;
                            }
                        }
                    }
                    else
                    {
                        error = "VIA Service call get Consults returned null";
                        _mdwsLog.ERROR_LEVEL = 2;
                    }
                }
                else
                {
                    error = null;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "getConsultsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        public List<KeyValuePair<string, string>> getHospitalAdmissionsFromVIA(out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedInpatientStayArray ret = null;

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                _loginSiteCode = this.TbiServiceTo.StationNumber;

                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");

                ret = _VIAService.getAdmissions(_bean);

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "getHospitalAdmissionsFromVIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { _loginSiteCode, _userID, _localPid });


                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.stays == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.stays orderby e.admitTimestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.admitTimestamp + ";H", e.location.name + " (" + DateTime.ParseExact(e.admitTimestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("MMM dd,yy") + ")")).ToList();
                            _mdwsLog.RETURNED_DATA = result.Serialize();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call get Hospital Admissions returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "getHospitalAdmissionsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        public List<KeyValuePair<string, string>> getVisitsFromVIA(DateTime fromDate, DateTime toDate, out string error)
        {
            taggedVisitArray ret = null;
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();

            string formattedFromDate = FormatDateForVIA(fromDate);
            string formattedToDate = FormatDateForVIA(toDate);

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                _loginSiteCode = this.TbiServiceTo.StationNumber;

                _bean.recordSiteCode = _loginSiteCode;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.startDate = formattedFromDate;
                _bean.endDate = formattedToDate;
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();

                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                ret = _VIAService.getVisits(_bean);

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "getVisitFromVIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { _loginSiteCode, _userID, _localPid });

                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        error = ret.fault.message;
                        result = null;

                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                    else
                    {
                        if (ret.count > 0)
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.visits orderby e.timestamp descending select new KeyValuePair<string, string>(e.location.id + ";" + e.timestamp + ";A", DateTime.ParseExact(e.timestamp, "yyyyMMdd.HHmmss", CultureInfo.InvariantCulture).ToString("g") + " " + e.location.name + " " + e.status)).ToList();
                            _mdwsLog.RETURNED_DATA = result.Serialize();
                        }
                        else
                        {
                            error = null;
                            _mdwsLog.ERROR_LEVEL = 1;
                        }
                    }
                }
                else
                {
                    error = "VIA Service call get Visits returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "getVisitsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// This will get the list of CPRS users for the correct VIA connection
        /// </summary>
        /// <param name="target">Beginning of first user name in the list</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> cprsUserLookupFromVIA(string target, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            userArray ret = null;

            try
            {
                queryBean _bean = new queryBean();
                provider _providerbean = new provider();

                target = TbiServiceTo.PatientName;
                _loginSiteCode = TbiServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.target = target;
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.provider = _providerbean;

                ret = _VIAService.cprsUserLookup(_bean);

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "cprsUserLookup-VIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { _loginSiteCode, _userID, _localPid });


                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.users == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.users orderby e.name select new KeyValuePair<string, string>(e.DUZ, e.name)).ToList();
                            _mdwsLog.RETURNED_DATA = result.Serialize();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = error = "VIA Service call CPRS User Lookup returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "cprsUserLookupFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        /// <summary>
        /// This will get the locations for the correct VIA connection
        /// </summary>
        /// <param name="filter">Use this to filter the results returned from MDWS</param>
        /// <param name="target">Purpose and expected values Unknown </param>
        /// <param name="direction">MDWS expects a value of 1 or -1</param>
        /// <returns></returns>
        public List<KeyValuePair<string, string>> getLocationsFromVIA(string filter, string target, string direction, out string error)
        {
            List<KeyValuePair<string, string>> result = new List<KeyValuePair<string, string>>();
            taggedHospitalLocationArray ret = null;
            string searchParameter = String.Empty;

            if (filter.Trim() == String.Empty)
            {
                searchParameter = target;
            }
            else
            {
                searchParameter = filter.Trim().ToUpper();
            }

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                _loginSiteCode = TbiServiceTo.StationNumber;
                _bean.target = searchParameter;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.direction = direction;
                _patientbean.localSiteId = _loginSiteCode;
                _providerbean.loginSiteCode = _loginSiteCode;
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                ret = _VIAService.getLocations(_bean);

                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "getLocations-VIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { _loginSiteCode, _userID, _localPid });


                if (ret != null)
                {
                    if (ret.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        error = ret.fault.message;
                    }
                    else
                    {
                        if (!(ret.count == 0 || ret.locations == null))
                        {
                            _mdwsLog.ERROR_LEVEL = 0;
                            error = "SUCCESS";
                            result = (from e in ret.locations where e.name.ToLower().StartsWith(filter.ToLower()) orderby e.name select new KeyValuePair<string, string>(e.id, e.name)).ToList();
                            _mdwsLog.RETURNED_DATA = result.Serialize();
                        }
                        else
                        {
                            _mdwsLog.ERROR_LEVEL = 1;
                            error = null;
                        }
                    }
                }
                else
                {
                    error = error = "VIA Service call get Locations returned null";
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "getLocationsFromVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        public noteResultTO writeNoteWithVIA(TBIServiceTOVIA tbiServiceTO)
        {
            noteResultTO _writeNoteResult = new noteResultTO();
            textTO _closeNoteResult = null;
            string titleIEN = "";
            string encounterString = "";
            string noteText = "";
            string authorDUZ = "";
            string cosignerDUZ = "";
            string consultIEN = "";
            string prfIEN = "";

            //set the values for the writeNote web service based on the noteType
            switch (tbiServiceTO.SelectedNoteType.Id)
            {
                case (int)NoteTypeEnum.ScheduledClinicAppointment:  //Scheduled Clinic Appointment

                    //if the note is for current visit or previous visit, call getVisits and have the user pick one . 
                    //Then encounterString = visit.location.id + ‘;’ + visit.timestamp + ‘;A’ 

                    //already formated from the original call to MDWS
                    encounterString = tbiServiceTO.SelectedVisitId;

                    break;
                case (int)NoteTypeEnum.HospitalAdmission:  //Hospital Admission

                    //if the note is for previous stay, call getAdmissions and have the user pick one. 
                    //Then encounterString = stay.location.id + ‘;’ + stay.admitTimestamp + ‘;H’ 

                    //already formated from the original call to MDWS
                    encounterString = tbiServiceTO.SelectedHospitalAdmissionId;


                    break;
                case (int)NoteTypeEnum.CurrentStay:  //Current Stay

                    //if the note is for current stay, encounterString = patient.locationId + ‘;’ + patient.admitTimestamp + ‘;H’ 

                    encounterString = tbiServiceTO.PatientLocationId + ";" + tbiServiceTO.PatientAdmintTimeStamp + ";H";

                    break;
                case (int)NoteTypeEnum.UnscheduledOrNewVisit: //Unscheduled or New Visit

                    if (!tbiServiceTO.isHistorical)
                    {
                        //if the note is for a new visit, but not hisorical, call getLocations and have the user pick one. 
                        //Then encounterString = location.id + ‘;’ + 20080515.1500 (today's date) + ‘;A’ 

                        encounterString = tbiServiceTO.SelectedLocationID + ";" + FormatDateForVIACPRS(DateTime.Now) + ";A";
                    }
                    else
                    {
                        //if the note is for a new visit, and is historical, 
                        //then previous step but end with ‘;E’ instead of ‘;A’. 

                        encounterString = tbiServiceTO.SelectedLocationID + ";" + FormatDateForVIACPRS(DateTime.Now) + ";E";

                    }

                    break;

            }

            //For preserving line breaks within a text block that is being written to a note, replace the \r\n or \n characters with a pipe (|).
            noteText = tbiServiceTO.NoteText.Replace("<br>", "|").Replace(Environment.NewLine, "|");

            authorDUZ = tbiServiceTO.DUZ;

            if (!String.IsNullOrEmpty(tbiServiceTO.SelectedConsultId))
            {
                consultIEN = tbiServiceTO.SelectedConsultId;
            }

            if (tbiServiceTO.SelectedNoteTitleId != "" && tbiServiceTO.SelectedNoteTitleId != null)
            {
                titleIEN = tbiServiceTO.SelectedNoteTitleId;
            }

            if (!String.IsNullOrEmpty(tbiServiceTO.SelectedCosignerID))
            {
                cosignerDUZ = tbiServiceTO.SelectedCosignerID;
            }

            prfIEN = "";

            bool savedFlag = false;


            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                _loginSiteCode = this.TbiServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _patientbean.localPid = HttpContext.Current.Session["ProviderDFN"].ToString();
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                _writeNoteResult = _VIAService.writeNote(titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN, _bean);
                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "writeNote-VIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { titleIEN, encounterString, noteText, authorDUZ, cosignerDUZ, consultIEN, prfIEN });

                if (_writeNoteResult != null)
                {
                    _mdwsLog.RETURNED_DATA = _writeNoteResult.Serialize();

                    if (!String.IsNullOrEmpty(_writeNoteResult.id))
                    {
                        mdwsLog.LogMDWSCall(_mdwsLog);
                        savedFlag = true;

                        CloseNote(titleIEN, consultIEN);
                    }

                    if (_writeNoteResult.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "writeNoteWithVIA()");
                _mdwsLog.ERROR_LEVEL = 2;
                throw;
            }

            if (!savedFlag)
                mdwsLog.LogMDWSCall(_mdwsLog);

            return _writeNoteResult;
        }

        private textTO CloseNote(string noteIEN, string consultIEN)
        {
            textTO result = null;

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();
                _loginSiteCode = this.TbiServiceTo.StationNumber;

                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _bean.recordSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.provider = _providerbean;

                result = _VIAService.closeNote(noteIEN, consultIEN, _bean);
                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "closeNote-VIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { noteIEN, consultIEN });

                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 1;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "CloseNote()");
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            return result;
        }

        public bool isCosignerRequired(TBIServiceTOVIA tbiServiceTO, out string errMsg)
        {
            bool ret = false;
            textTO result = null;
            string noteDefinitionIEN = "";
            string authorDUZ = "";
            //isCosignerRequired _isCosignerRequired1 = new isCosignerRequired();

            errMsg = "Service isCosignerRequired returned NULL value"; //Default error message

            noteDefinitionIEN = tbiServiceTO.SelectedNoteTitleId;
            authorDUZ = tbiServiceTO.DUZ;
            _loginSiteCode = tbiServiceTO.StationNumber;

            try
            {
                queryBean _bean = new queryBean();
                patient _patientbean = new patient();
                provider _providerbean = new provider();

                _bean.itemId = noteDefinitionIEN;
                _bean.recordSiteCode = _loginSiteCode;
                _bean.requestingApp = Helpers.GetDBConfigEntry("TBIServiceRequestingApp"); //"TBI";
                _bean.consumingAppToken = Helpers.GetDBConfigEntry("TBIServiceCred1");
                _bean.consumingAppPassword = Helpers.GetDBConfigEntry("TBIServiceCred2");
                _providerbean.name = HttpContext.Current.Session["ProviderName"].ToString();
                _providerbean.loginSiteCode = _loginSiteCode;
                _providerbean.userId = HttpContext.Current.Session["TokenizedDUZ"].ToString();
                _bean.criteria = authorDUZ;
                _patientbean.mpiPid = tbiServiceTO.PatientICN;

                _bean.patient = _patientbean;
                _bean.provider = _providerbean;

                result = _VIAService.isCosignerRequired(_bean);
                _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                _mdwsLog.SERVICE_NAME = "isCosignerRequired-VIA";
                _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { _loginSiteCode, authorDUZ });

                if (result != null)
                {
                    _mdwsLog.RETURNED_DATA = result.Serialize();

                    if (result.fault != null)
                    {
                        _mdwsLog.ERROR_LEVEL = 1;
                        errMsg = result.fault.message;
                    }
                    else
                    {
                        _mdwsLog.ERROR_LEVEL = 0;
                        errMsg = String.Empty;
                    }
                }
                else
                {
                    _mdwsLog.ERROR_LEVEL = 2;
                }
            }
            catch (Exception exc)
            {
                Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "isCosignerRequired()");
                Console.Write(exc.Message);
                _mdwsLog.ERROR_LEVEL = 2;
            }

            mdwsLog.LogMDWSCall(_mdwsLog);

            if (!String.IsNullOrEmpty(result.text))
            {
                switch (result.text.ToUpper())
                {
                    case "Y":
                        ret = true;
                        break;
                    case "N":
                        break;
                    default:
                        errMsg = "VIA Service isCosignerRequired returned uexpected value " + result.text;
                        break;
                }
            }

            return ret;
        }

        #endregion

        #region Private Helper Function

        private string FormatDateForVIA(DateTime dateValue)
        {
            // CPRS is expecting date in the following format "20000101.000000";
            return String.Format("{0:yyyyMMdd}", dateValue);
        }

        private string FormatDateForVIACPRS(DateTime dateValue)
        {
            DateTime _d = new DateTime();
            int _oddCalculationForVIA = 0;
            _oddCalculationForVIA = Convert.ToInt32(String.Format("{0:yyyy}", dateValue));
            // [ yyyy -1700]MMdd.HHmmss
            // VIA/CPRS is expecting date in the following format "20000101.000000";
            //yyyyMMddHHmmss
            return String.Format("{0:" + (_oddCalculationForVIA - 1700) + "MMdd.HHmmss}", dateValue);
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY>(e => e.SURVEY_RESULTs);
            lo.LoadWith<SURVEY_RESULT>(e => e.STD_QUESTION_CHOICE);
            lo.LoadWith<STD_QUESTION_CHOICE>(e => e.STD_QUESTION);
            lo.LoadWith<SURVEY>(e => e.STD_SURVEY_TYPE);
            lo.LoadWith<SURVEY>(e => e.SURVEY_NOTEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }


        private void SetSurveyLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY>(e => e.SURVEY_RESULTs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetQuestionnaireLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<SURVEY_RESULT>(e => e.SURVEY);
            lo.LoadWith<SURVEY_RESULT>(e => e.STD_QUESTION_CHOICE);
            lo.LoadWith<STD_QUESTION>(e => e.STD_QUESTION_CHOICEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private void SetQuestionsLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<STD_QUESTION>(e => e.STD_QUESTION_CHOICEs);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;

        }

        private IQueryable<SURVEY> LinqAll()
        {
            //populate LinqAll
            IQueryable<SURVEY> tempLinq = (from e in _db.SURVEYs
                                           select e);

            //TODO - add all business filtering rules 

            return tempLinq;

        }

        private IQueryable<SURVEY> SelectByPatientICNFilter(string ICN)
        {
            IQueryable<SURVEY> linqFilter = LinqAll();

            linqFilter = from t in linqFilter where t.patient_ICN == ICN select t;

            return linqFilter;
        }


        /// <summary>
        /// Get a survey by a surveyID.
        /// Return a survey object with all the survey results
        /// </summary>
        /// <param name="surveyId"></param>
        /// <returns></returns>
        public SURVEY GetSurveyWithAllAnswers(int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetSurveyLoadWith(_db);

                var survey = (from t in _db.SURVEYs
                              where t.SURVEYS_ID == surveyId
                              select t).FirstOrDefault();
                return survey;
            }
        }

        public List<SURVEY_RESULT> GetChoicesByQuestionNumber(string questionNumber, int surveyId)
        {
            using (_db = GetDataContext())
            {
                SetQuestionnaireLoadWith(_db);

                var choices = (from t in _db.SURVEY_RESULTs
                               where t.SURVEYS_ID == surveyId
                               where t.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER == questionNumber
                               select t).ToList();

                return choices;
            }

        }

        public string GetQuestionbyNumberAndSurveyType(string questionNumber, int surveyTypeId)
        {
            using (_db = GetDataContext())
            {
                string ret = "";

                SetQuestionnaireLoadWith(_db);

                var question = (from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE_ID == surveyTypeId
                                where t.QUESTION_NUMBER == questionNumber
                                select t).ToList();

                if (question.Count() > 0)
                {
                    ret = question.First().QUESTION_TEXT;
                }

                return ret;
            }

        }

        public bool IsInitialNoteType(string patientICN, int surveyType)
        {
            bool ret = false;

            //Get note type of latest completed survey
            string lastNoteType = GetLastCompletedNoteTypeByPatientICNAndSurveyType(patientICN, surveyType);

            if (String.IsNullOrEmpty(lastNoteType) ||
                String.Equals(lastNoteType, "Discharge", StringComparison.OrdinalIgnoreCase))
            {
                ret = true;
            }

            return ret;
        }

        public List<STD_QUESTION_CHOICE> GetMapiNoteTypeChoices()
        {
            return GetChoicesByQuestionNumberandSurveyType("-1", "MPAI");
        }

        public List<STD_QUESTION_CHOICE> GetMapiPersonReportingChoices()
        {
            return GetChoicesByQuestionNumberandSurveyType("0", "MPAI");
        }

        public TBI.InstrumentType GetInstrumentByID(int instrumentID)
        {
            return this.GetAllInstrumentList().FirstOrDefault(I => I.Id == instrumentID);
        }

        public List<STD_QUESTION_CHOICE> GetChoicesByQuestionNumberandSurveyType(string questionNumber, string surveyTypeCode)
        {
            using (_db = GetDataContext())
            {
                SetQuestionsLoadWith(_db);

                var question = (from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE.CODE == surveyTypeCode
                                where t.QUESTION_NUMBER == questionNumber
                                select t).ToList();

                var choices = (from t in _db.STD_QUESTION_CHOICEs
                               where t.STD_QUESTION_ID == question.FirstOrDefault().ID
                               where t.INACTIVE_FLAG == false
                               select t).OrderBy("CHOICE_SORT_ORDER").ToList();

                return choices;
            }
        }

        public IEnumerable<SURVEY> SelectByPatientICNAndSurveyType(string patientICN, int typeId, bool followUp)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.STD_SURVEY_TYPE_ID == typeId
                          select t;

                if (followUp)

                    surveys = from s in surveys where s.SURVEY_STATUS == "inprocess" select s;

                return surveys.ToList();
            }

        }

        public IEnumerable<SURVEY> SelectLastCompletedSurveyByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = SelectByPatientICNAndSurveyType(patientICN, typeId, false);
                surveys = (from s in surveys
                           where
                               String.Equals(s.SURVEY_STATUS, "completed", StringComparison.OrdinalIgnoreCase)
                           orderby s.UPDATED descending
                           select s).Take(1).ToList();


                return surveys.ToList();
            }
        }

        public IEnumerable<SURVEY> SelectByStatus(string statusId, string patientICN)
        {

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<SURVEY> surveys;

                surveys = from t in SelectByPatientICNFilter(patientICN)
                          where t.SURVEY_STATUS == statusId
                          select t;


                return surveys;
            }
        }

        public string GetLastCompletedNoteTypeByPatientICNAndSurveyType(string patientICN, int typeId)
        {
            string noteType = String.Empty;

            IEnumerable<SURVEY> survey;

            survey = SelectLastCompletedSurveyByPatientICNAndSurveyType(patientICN, typeId);

            if (survey.Count() == 1)
            {
                int surveyID = survey.First().SURVEYS_ID;

                try
                {
                    List<SURVEY_RESULT> lastCompletedSurveyResults = GetChoicesByQuestionNumber("-1", surveyID);

                    if (lastCompletedSurveyResults.Count() > 0)
                    {
                        noteType = lastCompletedSurveyResults.First().STD_QUESTION_CHOICE.CHOICE_NAME;
                    }
                }
                catch
                {
                    //Ignore error
                }
            }

            return noteType;
        }

        public int SelectByStatusCount(string statusId, string patientICN)
        {
            using (_db = GetDataContext())
            {
                return SelectByPatientICNFilter(patientICN).Where(c => c.SURVEY_STATUS == statusId).Count();
            }
        }

        public IEnumerable<STD_QUESTION> SelectAllQuestions(int typeId)
        {
            using (_db = GetDataContext())
            {
                SetQuestionnaireLoadWith(_db);

                var questions = from t in _db.STD_QUESTIONs
                                where t.STD_SURVEY_TYPE_ID == typeId && t.INACTIVE_FLAG == false
                                select t;

                return questions.ToList();
            }

        }
        public Int32 Update(SURVEY survey)
        {
            Int32 returnStatus = 0;

            if (TbiServiceTo != null)
            {
                survey.AUTHOR_DUZ = TbiServiceTo.DUZ;
                survey.PROVIDER_DUZ = TbiServiceTo.DUZ;
                if (!string.IsNullOrEmpty(TbiServiceTo.siteCode) && !string.IsNullOrEmpty(TbiServiceTo.ServerIP))
                {

                    survey.STD_INSTITUTION_ID = GetStdInstitutionId(_loginSiteCode.ToString(), TbiServiceTo.ServerIP);
                    if (!string.IsNullOrEmpty(TbiServiceTo.StationNumber))
                    {
                        UsersManager userManager = new UsersManager();
                        if (!string.IsNullOrEmpty(TbiServiceTo.SelectedCosignerID))
                        {
                            survey.PROVIDER_DUZ = TbiServiceTo.SelectedCosignerID;
                        }
                        survey.AUTHOR_ID = userManager.GetProviderIdByStaffIenAndSta3n(survey.AUTHOR_DUZ, VeteransAffairs.Registries.Business.Utilities.Helpers.ExtractSta3n(TbiServiceTo.StationNumber));
                        survey.PROVIDER_ID = userManager.GetProviderIdByStaffIenAndSta3n(survey.PROVIDER_DUZ, VeteransAffairs.Registries.Business.Utilities.Helpers.ExtractSta3n(TbiServiceTo.StationNumber));
                    }
                }

                using (_db = GetDataContext())
                {
                    _db.DeferredLoadingEnabled = false;


                    survey.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                        //create an instance of the custom eventArgs in order to populate the id selected
                        BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                        eventArgs.SavedItemId = survey.SURVEYS_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;

                        returnStatus = survey.SURVEYS_ID;

                        RaiseSaveEvent(this, eventArgs);

                    }
                    catch (Exception ex)
                    {
                        Helpers.HandleTBIException(ex, "VIAInstrumentManager.cs", "", "Update(SURVEY survey)");
                        Console.Write(ex.Message);
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                    }

                    return returnStatus;
                }
            }
            else
            {
                return 0;
            }
        }

        public System.Nullable<int> GetStdInstitutionId(string sta3n, string dataSourceProvider)
        {
            System.Nullable<int> result;
            int sta3nInteger = VeteransAffairs.Registries.Business.Utilities.Helpers.ExtractSta3n(sta3n);

            if (sta3nInteger != VeteransAffairs.Registries.Business.Utilities.Helpers.InvalidSta3n)
            {
                if (sta3nInteger > 900)
                { // test site sta3n
                    MDWSLog mdwsLog = new MDWSLog();
                    MDWS_LOG _mdwsLog = new MDWS_LOG();
                    // regionArray ret = null;

                    _cprsManager.SetMDWSLogDefaults(ref _mdwsLog);
                    _mdwsLog.SERVICE_NAME = "getVHA";
                    _mdwsLog.PARAMETERS = _cprsManager.xmlParameters(new string[] { dataSourceProvider });

                    try
                    {
                        //    if (_tbiService == null)
                        //    {
                        //        _tbiService = new EmrSvc();
                        //        _tbiService.CookieContainer = new System.Net.CookieContainer();
                        //    }
                        //    ret = _tbiService.getVHA();
                        //    if (ret != null)
                        //    {
                        //        _mdwsLog.RETURNED_DATA = ret.Serialize();

                        //        if (ret.fault != null)
                        //        {
                        //            _mdwsLog.ERROR_LEVEL = 1;
                        //        }
                        //        else
                        //        {
                        //            bool sta3nFound = false;
                        //            for (int i = 0; i < ret.regions.Length; i++)
                        //            {
                        //                RegionTO region = ret.regions[i];

                        //                for (int j = 0; j < region.sites.count; j++)
                        //                {
                        //                    SiteTO site = region.sites.sites[j];
                        //                    for (int k = 0; k < site.dataSources.count; k++)
                        //                    {
                        //                        DataSourceTO dataSource = site.dataSources.items[k];
                        //                        if (dataSource.provider.ToLower() == dataSourceProvider.ToLower())
                        //                        {
                        //                            sta3n = dataSource.siteId.tag;
                        //                            sta3nFound = true;
                        //                            break;
                        //                        }
                        //                    }

                        //                    if (sta3nFound)
                        //                    {
                        //                        break;
                        //                    }
                        //                }

                        //                if (sta3nFound)
                        //                {
                        //                    break;
                        //                }
                        //            }
                        //        }
                        //    }
                        //    else
                        //    {
                        //        _mdwsLog.ERROR_LEVEL = 2;
                        //    }
                    }
                    catch (Exception exc)
                    {
                        Helpers.HandleTBIException(exc, "VIAInstrumentManager.cs", "", "GetStdInstitutionId()");
                        _mdwsLog.ERROR_LEVEL = 2;
                        throw;
                    }

                    mdwsLog.LogMDWSCall(_mdwsLog);
                }


            }
            TbiServiceTo.StationNumber = sta3n;
            InstitutionManager institutionManager = new InstitutionManager();
            result = institutionManager.getInstitutionIDByStationNumber(sta3n);

            return result;

        }
        public void DeleteAllSurveyResults(SURVEY survey)
        {
            if (survey != null)
            {
                if (survey.SURVEY_RESULTs.Count > 0)
                {
                    survey.SetAsChangeTrackingRoot();

                    foreach (var result in survey.SURVEY_RESULTs)
                    {
                        result.SetAsDeleteOnSubmit();
                    }

                    using (_db = GetDataContext())
                    {
                        _db.DeferredLoadingEnabled = false;

                        survey.SynchroniseWithDataContext(_db); //this line traverses all entities, attaching all of them as appropriate to the data context.

                        try
                        {
                            _db.SubmitChanges(ConflictMode.ContinueOnConflict);

                            //create an instance of the custom eventArgs in order to populate the id selected
                            BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();
                            eventArgs.SavedItemId = survey.SURVEYS_ID;
                            eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                        }
                        catch
                        {
                            _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);

                        }
                    }
                }
            }
        }

        /// <summary>
        /// Accepts a List<STD_SURVEY_TYPE> objects, converts and returns the equivalent List<InstrumentType>
        /// </summary>
        /// <returns></returns>
        private List<TBI.InstrumentType> convertStdSurveyTypeListToInstrumentType(List<STD_SURVEY_TYPE> surveyTypes)
        {
            List<TBI.InstrumentType> returnList = new List<TBI.InstrumentType>();

            foreach (STD_SURVEY_TYPE survey in surveyTypes)
            {
                // for Mayo-Portland Adaptability Inventory-4 Participation Index and TBI/Polytrauma Individualized Rehabilitation/Reintegration Plan of Care - custom titles required

                TBI.InstrumentType instrumentType = new TBI.InstrumentType();
                instrumentType.Id = survey.ID;
                instrumentType.Name = survey.NAME;
                instrumentType.IsActive = survey.IS_ACTIVE;

                if (survey.CODE == "MPAI")
                {
                    List<string> titleMpai = new List<string>();
                    titleMpai.Add("Mayo-Portland Adaptability Inventory-4");
                    titleMpai.Add("Participation Index (M2PI)");
                    titleMpai.Add("Muriel D. Lezak, PhD, ABPP & James F. Malec, PhD, ABPP");
                    titleMpai.Add("");
                    titleMpai.Add("Used as VA Interdisciplinary Team Assessment of Community Functioning");

                    instrumentType.Title = titleMpai;

                }
                else if (survey.CODE == "CAREPLAN")
                {
                    List<string> titleCarePlan = new List<string>();
                    titleCarePlan.Add("TBI/Polytrauma Individualized Rehabilitation/Reintegration Plan of Care");

                    instrumentType.Title = titleCarePlan;
                }
                else
                {
                    instrumentType.Title = new List<string>();
                    instrumentType.Title.Add(survey.NAME);
                }

                returnList.Add(instrumentType);
            }

            return returnList;
        }

        #endregion

        #region Serialized TBI Classes

        [Serializable()]
        public class TBIServiceTOVIA
        {
            public TBIServiceTOVIA()
            {
                List<string> titleMpai = new List<string>();
                titleMpai.Add("Mayo-Portland Adaptability Inventory-4");
                titleMpai.Add("Participation Index (M2PI)");
                titleMpai.Add("Muriel D. Lezak, PhD, ABPP & James F. Malec, PhD, ABPP");
                titleMpai.Add("");
                titleMpai.Add("Used as VA Interdisciplinary Team Assessment of Community Functioning");

                List<string> titleCarePlan = new List<string>();
                titleCarePlan.Add("TBI/Polytrauma Individualized Rehabilitation/Reintegration Plan of Care");


                // Title for L - Test
                List<string> medredLTest = new List<string>();
                medredLTest.Add("L - Test");

                // Title for Locomotor Capability Index - 5 (LCI - 5)
                List<string> medredLCIv2 = new List<string>();
                medredLCIv2.Add("Locomotor Capability Index - 5 (LCI - 5)");

                // Title for 2 Minute Walk Test
                List<string> medredTwoMinWalkTest = new List<string>();
                medredTwoMinWalkTest.Add("2 Minute Walk Test");

                // Title for Patient Global Impression of Change (PGIC)
                List<string> pgic = new List<string>();
                pgic.Add("Patient Global Impression of Change (PGIC)");

                // Title for Neurobehavioral Symptom Inventory (NSI)
                List<string> nsi = new List<string>();
                nsi.Add("Neurobehavioral Symptom Inventory (NSI)");

                // Title for Disability Rating Scale (DRS)
                List<string> drs = new List<string>();
                drs.Add("Disability Rating Scale (DRS)");

                // Title for QUEST
                List<string> quest = new List<string>();
                quest.Add("Quebec User Evaluation of Satisfaction with Assistive Technology (QUEST)");

                // Title for Satisfaction with Life Scale (SWLS)
                List<string> satisfactionWithLife = new List<string>();
                satisfactionWithLife.Add("Satisfaction with Life Scale (SWLS)");

                // Title for Functional Mobility Assessment (FMA) 
                List<string> functionalMobilityAssessment = new List<string>();
                functionalMobilityAssessment.Add("Functional Mobility Assessment (FMA)");

                // Title for Berg Balance Scale
                List<string> bergBalanceScale = new List<string>();
                bergBalanceScale.Add("Berg Balance Scale");

                // Title for VA Low Vision Visual Functioning (VA LV VFQ 20) Survey
                List<string> vaLowVision = new List<string>();
                vaLowVision.Add("VA Low Vision Visual Functioning (LA LV VFQ 20) Survey");

                Instruments = new List<InstrumentType>() 
            { 
                new InstrumentType { Id = 2, Name = "COMPREHENSIVE TBI EVALUATION" } ,
                new InstrumentType { Id = 3, Name = "TBI FOLLOW-UP ASSESSMENT" } ,
                new InstrumentType { Id = 4, Name = "MPAI-4 PARTICIPATION INDEX (M2PI)", Title = titleMpai } ,
                new InstrumentType { Id = 5, Name = "REHABILITATION AND REINTEGRATION PLAN", Title = titleCarePlan }, 
               new InstrumentType { Id = 7, Name = "L - Test", Title =  medredLTest }, 
                new InstrumentType { Id = 8, Name = "Locomotor Capability Index - 5 (LCI - 5)", Title =  medredLCIv2},
                new InstrumentType { Id = 9, Name = "2 Minute Walk Test", Title =  medredTwoMinWalkTest},
                new InstrumentType { Id = 10, Name = "Patient Global Impression of Change (PGIC)", Title = pgic},
                new InstrumentType { Id = 11, Name = "Neurobehavioral Symptom Inventory (NSI)", Title = nsi }, 
                new InstrumentType { Id = 12, Name = "Disability Rating Scale (DRS)", Title = drs }, 
                new InstrumentType { Id = 13, Name = "Quebec User Evaluation of Satisfaction with Assistive Technology (QUEST)", Title = quest },
                new InstrumentType { Id = 14, Name = "Satisfaction with Life Scale (SWLS)", Title = satisfactionWithLife }, 
                new InstrumentType { Id = 15, Name = "Functional Mobility Assessment (FMA)", Title = functionalMobilityAssessment }, 
                new InstrumentType { Id = 16, Name = "Berg Balance Scale", Title = bergBalanceScale },
                new InstrumentType { Id = 17, Name = "VA Low Vision Visual Functioning (LA LV VFQ 20) Survey", Title = vaLowVision }

            };
            }

            public string siteCode { get; set; }
            public string DUZ { get; set; }
            public string DFN { get; set; }

            public string ServerIP { get; set; }
            public string ServerPort { get; set; }

            public string UserName { get; set; }
            public string PatientName { get; set; }
            public string PatientDOB { get; set; }
            public string PatientGender { get; set; }
            public string PatientSSN { get; set; }
            public string PatientLocationId { get; set; }
            public string PatientICN { get; set; }
            public string PatientDFN { get; set; }
            public string STA6N { get; set; }
            public string PatientAdmintTimeStamp { get; set; }
            public string InstitutionName { get; set; }


            public int CprsSentTBIEvaluationNoteId { get; set; }
            public int CprsSentTBIFollowUpNoteId { get; set; }

            public NoteType SelectedNoteType { get; set; }
            public string NoteText { get; set; }

            public TBI.InstrumentType SelectedInstrumentType { get; set; }

            public string SelectedNoteTitleId { get; set; }
            public string SelectedConsultId { get; set; }
            public string SelectedVisitId { get; set; }
            public string SelectedHospitalAdmissionId { get; set; }
            public string SelectedLocationID { get; set; }
            public string SelectedNewVisitId { get; set; }

            public string SelectedCosignerID { get; set; }

            public bool isHistorical { get; set; }

            public List<InstrumentType> Instruments;

            public string StationNumber { get; set; }

        }

        public enum NoteTypeEnum : int
        {
            ScheduledClinicAppointment = 0,
            HospitalAdmission = 1,
            CurrentStay = 2,
            UnscheduledOrNewVisit = 3
        }

        [Serializable()]
        public class NoteType
        {
            public int Id { get; set; }
            public string Name { get; set; }
        }

        [Serializable()]
        public class InstrumentType
        {
            public int Id { get; set; }
            public string Name { get; set; }
            public string Status { get; set; }
            public List<string> Title { get; set; }
            public Boolean IsActive { get; set; }
        }

        #endregion

    }
}